/*
// Error.cpp
//
// AUTHORS:     Julius Pettersson
// CREDITS:     n/a
// LCHANGE:     22 01 12 13:44
// PURPOSE:     error reporting and logging
// LICENSE:     Expat/MIT License, see `copying.txt'
//
*/

// standard headers
#include <array>
#include <cstdarg>
#include <ctime>
#include <fstream>
#include <sstream>
#include <string>

// miscellaneous headers
#include <windows.h>

#include "Error.hpp"

// default filename for error logging file
#define ERRLOG_FILENAME     "error_log.txt"

#define TEMPWHY_MAXALLOC    256

namespace error {

/*
// Outputs error message to UI and log file.
*/
void log(const char *ppFile, int ppLine, const std::string &what, const std::string &why, ...)
{
    const std::time_t rawTime_  = std::time(nullptr); // seconds since 01-01-1970
    std::string what_           = "(TOP SECRET)";
    std::string why_            = "(The programmer decided you don't need to know more.)";
    std::string windowTitle_;
    std::string windowMessage_;
    std::string ppFile_         = "(Unknown file)";

    // did the programmer specify the bugged unit?
    if (!what.empty())
        what_ = what;

    if (ppFile != nullptr && *ppFile != '\0')
        ppFile_ = ppFile;

    // TODO: use std::to_string() when available
    what_ = what_ + ", " + ppFile_ + " @ " + static_cast<std::ostringstream *> (&(std::ostringstream() << ppLine))->str();

    if (!why.empty())
    {
        // build message from parameters
        std::va_list args_;
        std::array<char, TEMPWHY_MAXALLOC> tempWhy_;

        va_start(args_, why);
        std::vsnprintf(&tempWhy_.front(), tempWhy_.size(), why.c_str(), args_);
        va_end(args_);

        why_ = tempWhy_.data();
    }

    std::ofstream errorLogFile_(ERRLOG_FILENAME, std::ofstream::app);

    if (errorLogFile_.is_open())
    {
        // TODO: check for exceptions
        errorLogFile_
            << " WHEN --- " << std::ctime(&rawTime_)
            << " WHAT --- " << what_ << '\n'
            << "  WHY --- " << why_ << "\n\n";

        errorLogFile_.close();

        // could save successfully to errorlog file
        windowMessage_ = why_ + "\n\nThis error was saved to file `" + ERRLOG_FILENAME + "'.\nPress Space to continue...";
    }
    else // couldn't save successfully to errorlog file
        windowMessage_ = why_ + "\n\nAlso, could not open file `" + ERRLOG_FILENAME + "' to save this error.\nPress Space to continue...";

    windowTitle_ = "Error: " + what_;
    MessageBox(nullptr, windowMessage_.c_str(), windowTitle_.c_str(), MB_OK | MB_ICONEXCLAMATION);
}

} /* namespace error */
